#include "TusTssSony.h"
#include "SignInSony.h"
#include "UserProfileSony.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"

using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{
	CachedTusTss gTusTss;

	DO_EXPORT( bool, PrxTUSIsBusy ) ()
	{
		return gTusTss.IsTusBusy();
	}

	DO_EXPORT( bool, PrxTusTssGetLastError ) (ResultCode* result)
	{
		return gTusTss.GetLastError(result);
	}

	DO_EXPORT( ErrorCode, PrxTUSClearVariablesToSet ) ()
	{
		return gTusTss.ClearTusVariablesToSet();
	}

	DO_EXPORT( ErrorCode, PrxTUSAddVariableToSet ) (int slot, Int64 value)
	{
		return gTusTss.AddTusVariablesToSet(slot, value);
	}

	DO_EXPORT( ErrorCode, PrxTUSClearVariablesToGet ) ()
	{
		return gTusTss.ClearTusVariablesToGet();
	}

	DO_EXPORT( ErrorCode, PrxTUSAddVariableToGet ) (int slot)
	{
		return gTusTss.AddTusVariablesToGet(slot);
	}

	DO_EXPORT( ErrorCode, PrxTUSRequestVariables ) ()
	{
		return gTusTss.RequestTusVariables(NULL, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSSetVariables ) ()
	{
		return gTusTss.SetTusVariables(NULL, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSSetVariablesForUser ) (const unsigned char* npID)
	{
		return gTusTss.SetTusVariables(npID, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSRequestVariablesForUser ) (const unsigned char* npID)
	{
		return gTusTss.RequestTusVariables(npID, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSSetVariablesForVirtualUser ) (const unsigned char* onlineID)
	{
		return gTusTss.SetTusVariables(onlineID, true);
	}

	DO_EXPORT( ErrorCode, PrxTUSModifyVariables) ()
	{
		return gTusTss.ModifyTusVariables(NULL, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSModifyVariablesForUser )(const unsigned char* npID)
	{
		return gTusTss.ModifyTusVariables(npID, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSModifyVariablesForVirtualUser )(const unsigned char* onlineID)
	{
		return gTusTss.ModifyTusVariables(onlineID, true);
	}


	DO_EXPORT( ErrorCode, PrxTUSRequestVariablesForVirtualUser ) (const unsigned char* onlineID)
	{
		return gTusTss.RequestTusVariables(onlineID, true);
	}

	DO_EXPORT( int, PrxTUSGetVariableCount ) ()
	{
		return gTusTss.GetTusVariableCount();
	}

	DO_EXPORT( Int64, PrxTUSGetVariableValue ) (int index)
	{
		return gTusTss.GetTusVariableValue(index);
	}

	DO_EXPORT( ErrorCode, PrxTUSGetVariable ) (int index, TusRetrievedVar* result)
	{
		return gTusTss.GetTusVariable(index, result);
	}

	DO_EXPORT( ErrorCode, PrxTUSSetData ) (int slotId, void* data, int dataSize)
	{
		return gTusTss.SetTusData(slotId, data, dataSize, NULL, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSSetDataForUser ) (int slotId, void* data, int dataSize, const unsigned char* npID)
	{
		return gTusTss.SetTusData(slotId, data, dataSize, npID, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSSetDataForVirtualUser ) (int slotId, void* data, int dataSize, const unsigned char* onlineID)
	{
		return gTusTss.SetTusData(slotId, data, dataSize, onlineID, true);
	}

	DO_EXPORT( ErrorCode, PrxTUSRequestData ) (int slotID)
	{
		return gTusTss.RequestTusData(slotID, NULL, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSRequestDataForUser ) (int slotID, const unsigned char* npID)
	{
		return gTusTss.RequestTusData(slotID, npID, false);
	}

	DO_EXPORT( ErrorCode, PrxTUSRequestDataForVirtualUser ) (int slotID, const unsigned char* onlineID)
	{
		return gTusTss.RequestTusData(slotID, onlineID, true);
	}

	DO_EXPORT( ErrorCode, PrxTUSGetData ) (TusTssData* data)
	{
		return gTusTss.GetTusData(data);
	}

	DO_EXPORT( bool, PrxTSSIsBusy ) ()
	{
		return gTusTss.IsTssBusy();
	}

	DO_EXPORT( ErrorCode, PrxTSSRequestData ) ()
	{
		return gTusTss.RequestTssData();
	}

	DO_EXPORT( ErrorCode, PrxTSSRequestDataFromSlot ) (int slot)
	{
		return gTusTss.RequestTssDataFromSlot(slot);
	}

	DO_EXPORT( ErrorCode, PrxTSSGetData ) (TusTssData* data)
	{
		return gTusTss.GetTssData(data);
	}

	CachedTusTss::CachedTusTss()
		: m_Busy(false)
		, m_LastResult("TusTss")
		, m_TssBuffer(NULL)
	{
	}

	CachedTusTss::~CachedTusTss()
	{
		free(m_TssBuffer);
	}

	bool CachedTusTss::ProcessEvent(const sce::Toolkit::NP::Event& event)
	{
		Mutex::AutoLock lock(m_Lock);
		bool handled = false;

		switch(event.event)
		{
			case Event::tusDataSet:							// An event generated when data is uploaded to a TUS (title user storage) server.
				if(event.returnCode != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				}
				else
				{
					UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TUSDataSet);
				}
				m_Busy = false;
				handled = true;
				break;

			case Event::tusVariablesSet:					// An event generated when variables are set on a TUS (title user storage) server.
				if(event.returnCode != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				}
				else
				{
					if(m_ModifyState.isModifying)
					{
						UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TUSVariablesModified);
						m_ModifyState.Reset();
					}
					else
					{	
						UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TUSVariablesSet);
					}
				}
				m_Busy = false;
				handled = true;
				break;

			case Event::tusDataReceived:					// An event generated when data is retrieved from a TUS (title user storage) server.
				if(event.returnCode != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				}
				else
				{
					UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TUSDataReceived);
				}
				m_Busy = false;
				handled = true;
				break;

			case Event::tusVariablesReceived:				// An event generated when variables are retrieved from a TUS (title user storage) server.
				if(event.returnCode != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				}
				else if(!m_ModifyState.isModifying)
				{
					UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TUSVariablesReceived);
				}

				if(m_ModifyState.isModifying)
				{
					// Add current values to requested values.
					if(m_TusVarsIn.hasResult())
					{
						std::vector<SceNpTusVariable>& variables = *m_TusVarsIn.get();
						for(int i=0; i<variables.size(); i++)
						{
							m_VariablesToSet[i].value += variables[i].variable;
							variables[i].variable = m_VariablesToSet[i].value;
						}

						// Set the new values.
						m_Busy = false;
						SetTusVariables((const unsigned char*)&m_ModifyState.npID.handle, m_ModifyState.isVirtualUser);
					}
					else
					{
						m_LastResult.SetResult(NP_ERR_TUS_NO_VAR_DATA, true);
						Messages::AddMessage(Messages::kNPToolKit_TusTssError);
						m_Busy = false;
					}
				}
				else
				{
					m_Busy = false;
				}

				handled = true;
				break;

			case Event::tusError:							// An event generated when an error occurs while working with a TUS (title user storage) server.
				m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				m_ModifyState.Reset();
				m_Busy = false;
				handled = true;
				break;

			case Event::tusCrossSaveDataSet:				// An event generated when data is uploaded to a TUS (title user storage) server for cross-saves.
			case Event::tusCrossSaveDataReceived:			// An event generated when data is retrieved from a TUS (title user storage) server for cross-saves.
				if(event.returnCode != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				}

				UnityPlugin::Messages::LogWarning("TUS event not handled: event=%d\n", event.event);
				m_Busy = false;
				handled = true;
				break;
				
			case Event::tssGotData:							// An event generated when data has been retrieved from a TSS (title small storage) server.
				if(event.returnCode != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				}
				else
				{
					UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TSSDataReceived);
				}
				m_Busy = false;
				handled = true;
				break;

			case Event::tssNoData:							// An event generated when no data is found on the TSS (title small storage) server.
				if(event.returnCode != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				}
				else
				{
					UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_TSSNoData);
				}
				m_Busy = false;
				handled = true;
				break;


			case Event::tssError:							// An event generated when an error occurs while working with a TSS (title small storage) server.
				if(event.returnCode != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				}
				m_Busy = false;
				handled = true;
				break;

			default:
				if(event.returnCode != SCE_TOOLKIT_NP_SUCCESS)
				{
					m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
					Messages::AddMessage(Messages::kNPToolKit_TusTssError);
				}
				UnityPlugin::Messages::LogWarning("Unexpected event from TUS service: event=%d\n", event.event);
				handled = true;
				break;
		}

		return handled;
	}

	bool CachedTusTss::IsTusBusy()
	{
		Mutex::AutoLock lock(m_Lock);
		return m_Busy;
	}

	ErrorCode CachedTusTss::ClearTusVariablesToSet()
	{
		m_VariablesToSet.clear();
		return NP_OK;
	}

	ErrorCode CachedTusTss::AddTusVariablesToSet(int slot, Int64 value)
	{
		Variable variable(slot, value);
		m_VariablesToSet.push_back(variable);
		return NP_OK;
	}

	ErrorCode CachedTusTss::SetTusVariables(const unsigned char* npID, bool isVirtualUser)
	{
		if(IsTusBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		int ret;
		TusSetVarsInputParams params;

		if(npID)
		{
			memcpy(&params.npid, npID, sizeof(SceNpId));
		}
		else
		{
			Future<SceNpId> npid;
			ret = sce::Toolkit::NP::UserProfile::Interface::getNpId(&npid, false);
			if (ret < 0)
			{
				return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			}

			params.npid = *npid.get();
			if(isVirtualUser)
			{
				// Virtual user not allowed with default npID.
				return m_LastResult.SetResult(NP_ERR_FAILED, true, __FUNCTION__, __LINE__);
			}
		}


		for(unsigned int i=0; i<m_VariablesToSet.size(); i++)
		{
			params.vars.push_back(TusVariable(m_VariablesToSet[i].slotId, m_VariablesToSet[i].value));
		}
		params.isVirtualUser = isVirtualUser;

		ret = TUS::Interface::setVariables(params, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		ClearTusVariablesToSet();

		m_Busy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedTusTss::ModifyTusVariables(const unsigned char* npID, bool isVirtualUser)
	{
		if(IsTusBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		// First step is to get the variables.

		m_LastResult.Reset();
		m_TusVarsIn.reset();

		int ret;

		TusGetVarsInputParams params;
		memset(&params, 0, sizeof(params));

		if(npID)
		{
			memcpy(&params.npid, npID, sizeof(SceNpId));
		}
		else
		{
			Future<SceNpId> npid;
			ret = sce::Toolkit::NP::UserProfile::Interface::getNpId(&npid, false);
			if (ret < 0)
			{
				return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			}

			params.npid = *npid.get();
			if(isVirtualUser)
			{
				// Virtual user not allowed with default npID.
				return m_LastResult.SetResult(NP_ERR_FAILED, true, __FUNCTION__, __LINE__);
			}
		}


		// use m_VariablesToSet as we need slot ID's and values.
		int slotCount = std::min(16, (int)m_VariablesToSet.size());	// Max of 16 slots, which = SCE_TOOLKIT_NP_TUS_MAX_SLOTS on PS4.
		int slotIDs[16];
		for(int i=0; i<slotCount; i++)
		{
			slotIDs[i] = m_VariablesToSet[i].slotId;
		}

		params.slotIds = &slotIDs[0];
		params.numSlots = slotCount;
		params.isVirtualUser = isVirtualUser;

		ret = TUS::Interface::getVariables(&m_TusVarsIn, params, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		// Setup the modify state.
		m_ModifyState.isModifying = true;
		m_ModifyState.isVirtualUser = isVirtualUser;
		memcpy(&m_ModifyState.npID, &params.npid, sizeof(SceNpId));

		m_Busy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedTusTss::ClearTusVariablesToGet()
	{
		m_VariablesToGet.clear();
		return NP_OK;
	}
	
	ErrorCode CachedTusTss::AddTusVariablesToGet(int slot)
	{
		m_VariablesToGet.push_back(slot);
		return NP_OK;
	}

	ErrorCode CachedTusTss::RequestTusVariables(const unsigned char* npID, bool isVirtualUser)
	{
		if(IsTusBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		m_ModifyState.Reset();
		m_LastResult.Reset();
		m_TusVarsIn.reset();

		int ret;

		TusGetVarsInputParams params;
		memset(&params, 0, sizeof(params));

		if(npID)
		{
			memcpy(&params.npid, npID, sizeof(SceNpId));
		}
		else
		{
			Future<SceNpId> npid;
			ret = sce::Toolkit::NP::UserProfile::Interface::getNpId(&npid, false);
			if (ret < 0)
			{
				return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			}

			params.npid = *npid.get();
			if(isVirtualUser)
			{
				// Virtual user not allowed with default npID.
				return m_LastResult.SetResult(NP_ERR_FAILED, true, __FUNCTION__, __LINE__);
			}
		}


		params.slotIds = &m_VariablesToGet[0];
		params.numSlots = m_VariablesToGet.size();
		params.isVirtualUser = isVirtualUser;

		ret = TUS::Interface::getVariables(&m_TusVarsIn, params, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}

	int CachedTusTss::GetTusVariableCount()
	{
		if(IsTusBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		if(m_TusVarsIn.hasResult())
		{
			return m_TusVarsIn.get()->size();
		}

		return 0;
	}

	Int64 CachedTusTss::GetTusVariableValue(int index)
	{
		if(IsTusBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		if(m_TusVarsIn.hasResult())
		{
			const std::vector<SceNpTusVariable>& variable = *m_TusVarsIn.get();
			return variable[index].variable;
		}
		else
		{
			m_LastResult.SetResult(NP_ERR_TUS_NO_VAR_DATA, true);
		}

		return 0;
	}

	ErrorCode CachedTusTss::GetTusVariable(int index, TusRetrievedVar* result)
	{
		if(IsTusBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		if(m_TusVarsIn.hasResult())
		{
			const std::vector<SceNpTusVariable>& varArray = *m_TusVarsIn.get();
			const SceNpTusVariable& variable = varArray[index];
			result->hasData = variable.hasData;
			result->variable = variable.variable;
			result->oldVariable = variable.oldVariable;
			result->lastChangedDate = variable.lastChangedDate.tick * 10;
			result->lastChangeAuthorNpID = (const unsigned char*)&variable.lastChangedAuthorId.handle;
			result->ownerNpID = (const unsigned char*)&variable.ownerId.handle;
			result->npIDSize = sizeof(SceNpId);
		}
		else
		{
			m_LastResult.SetResult(NP_ERR_TUS_NO_VAR_DATA, true);
		}

		return m_LastResult.GetResult();
	}

	ErrorCode CachedTusTss::SetTusData(int slotId, void* data, int dataSize, const unsigned char* npID, bool isVirtualUser)
	{
		if(IsTusBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		int ret;
		TusSetDataInputParams params;
		memset(&params, 0, sizeof(params));
		params.slotId = slotId;

		if(npID)
		{
			memcpy(&params.npid, npID, sizeof(SceNpId));
		}
		else
		{
			Future<SceNpId> npid;
			ret = sce::Toolkit::NP::UserProfile::Interface::getNpId(&npid, false);
			if (ret < 0)
			{
				return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			}

			params.npid = *npid.get();
			if(isVirtualUser)
			{
				// Virtual user not allowed with default npID.
				return m_LastResult.SetResult(NP_ERR_FAILED, true, __FUNCTION__, __LINE__);
			}
		}


		params.data.buffer = data;
		params.data.bufferSize = dataSize;
		params.isVirtualUser = isVirtualUser;

		ret = TUS::Interface::setData(params, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedTusTss::RequestTusData(int slotID, const unsigned char* npID, bool isVirtualUser)
	{
		if(IsTusBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		int ret;
		m_TusDataIn.reset();

		TusGetDataInputParams params;
		memset(&params, 0, sizeof(params));

		if(npID)
		{
			memcpy(&params.npid, npID, sizeof(SceNpId));
		}
		else
		{
			Future<SceNpId> npid;
			ret = sce::Toolkit::NP::UserProfile::Interface::getNpId(&npid, false);
			if (ret < 0)
			{
				return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			}
			params.npid = *npid.get();
			if(isVirtualUser)
			{
				// Virtual user not allowed with default npID.
				return m_LastResult.SetResult(NP_ERR_FAILED, true, __FUNCTION__, __LINE__);
			}
		}


		params.slotId = slotID;
		params.isVirtualUser = isVirtualUser;

		ret = TUS::Interface::getData(&m_TusDataIn, params, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedTusTss::GetTusData(TusTssData* data)
	{
		if(IsTusBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		data->dataSize = m_TusDataIn.get()->data.bufferSize;
		data->data = m_TusDataIn.get()->data.buffer;
		return m_LastResult.GetResult();
	}

	bool CachedTusTss::IsTssBusy()
	{
		Mutex::AutoLock lock(m_Lock);
		return m_Busy;
	}

	ErrorCode CachedTusTss::RequestTssData()
	{
		if(IsTssBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		int ret = TSS::Interface::getData(&m_TssDataIn, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
	}

	ErrorCode CachedTusTss::RequestTssDataFromSlot(int /*slotID*/)
	{
		if(IsTssBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);
		
		m_LastResult.Reset();

#if(1)	// TODO: Currently the Vita SCE npToolkit doesn't implement TSS::Interface::GetDataStatus or TSS::Interface::getDataFromSlot despite what the docs say.
		return m_LastResult.GetResult();
#else
		Future<SceNpTssDataStatus> status;

		int ret = TSS::Interface::getDataStatus(&status, slotID, false);
		if(ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		free(m_TssBuffer);
		m_TssBufferSize = status.get()->contentLength;
		m_TssBuffer = malloc(m_TssBufferSize);

		TssInputParams params;
		memset(&params, 0, sizeof(params));
		params.buffer = m_TssBuffer;
		params.size = m_TssBufferSize;
		params.slotId = slotID;
		ret = TSS::Interface::getDataFromSlot(&m_TssDataIn, params, true);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}

		m_Busy = true;
		return m_LastResult.GetResult();
#endif
	}

	ErrorCode CachedTusTss::GetTssData(TusTssData* data)
	{
		if(IsTssBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		Mutex::AutoLock lock(m_Lock);

		m_LastResult.Reset();
		data->dataSize = m_TssDataIn.get()->size;
		data->data = m_TssDataIn.get()->buffer;
		// TODO: m_TssDataIn.get()->status?
		return m_LastResult.GetResult();
	}
}
